#!/bin/bash
#
# Copyright (c) 2006 Mellanox Technologies. All rights reserved.
# 
# This Software is licensed under one of the following licenses:
#
# 1) under the terms of the "Common Public License 1.0" a copy of which is
#    available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/cpl.php.
#
# 2) under the terms of the "The BSD License" a copy of which is
#    available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/bsd-license.php.
#
# 3) under the terms of the "GNU General Public License (GPL) Version 2" a
#    copy of which is available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/gpl-license.php.
#
# Licensee has the right to choose one of the above licenses.
#
# Redistributions of source code must retain the above copyright
# notice and one of the license notices.
#
# Redistributions in binary form must reproduce both the above copyright
# notice, one of the license notices in the documentation
# and/or other materials provided with the distribution.
#
#
#  $Id: install.sh 9099 2006-08-24 12:37:33Z vlad $
# 

usage()
{
cat << EOF

    Usage: `basename $0` [--prefix PREFIX] [--batch] [--help] [--tmpdir DIR]
                         [-kver <kernel version>] [-k|--ksrc DIR]
                         [--memtrack]
                         [--nosu] [--clean]
                         [--packages module1 module2 ... -- | --all]

                         [--with-all-libs]   : include all user level libraries (all above --with-...)
                         [--debug]  : enable debug option for IB modules
                         
                         [--build_root BUILD_ROOT]

                         [--without-user]    : do not install userspace libraries/binaries
                         [--without-kernel]  : do not install IB kernel modules
                         Available Packages: ib_verbs ib_mthca ib_ipoib ib_sdp ib_srp ib_iser ib_rds ib_ipath

                         Install script supports configure options
                                                              
EOF
                         $(dirname $0)/*user/configure --help
                         $(dirname $0)/*kernel/configure --help
}


###############################################################################
#                          Default values assigment                           #
###############################################################################

# remove any aliases
unalias -a
# Find the directory where this script resides
installdir=$(cd ${0%*/*};pwd)
installdir_user=$(readlink -f ${installdir}/*user)
installdir_kernel=$(readlink -f ${installdir}/*kernel)
prog=$0
package_name=$(echo ${installdir##*/} | cut -d '-' -f 1 | tr -d '[:space:]')
package_name_user=$(echo ${installdir_user##*/} | cut -d '-' -f 1 | tr -d '[:space:]')
package_name_kernel=$(echo ${installdir_kernel##*/} | cut -d '-' -f 1 | tr -d '[:space:]')
tmpdir=/tmp/openib_gen2
prefix=/usr/local/ofed
clean=no
batch=no
nosu=no
SET_ENV=1
MODPROBE_UPDATE=1
PACKAGES=""
ALL_PACKAGES="ib_verbs ib_mthca ib_ipoib ib_sdp ib_srp rds" # ib_iser ib_ipath ib_rds 
ARCH=`uname -m`
BUILD_MODULES=1
WITH_PACKAGES_USER=""
WITH_PACKAGES_KERNEL=""
CONFIG_PARAMS=""
WITH_ALLPACKAGES="--with-libibverbs --with-libmthca --with-libibcm --with-libsdp --with-librdmacm --with-management-libs --with-opensm --with-openib-diags --with-perftest --with-mstflint --with-srptools --with-ipoibtools"

NULL=/dev/null
IB_CONF_DIR=/etc/infiniband
BUILD_ROOT=""
QUILT_PATCH=1
LIBSDP=0
INSTALL_KERNEL=1
INSTALL_USER=1

if [ -f /etc/redhat-release ]; then
        DISTRIBUTION="RedHat"
elif [ -f /etc/rocks-release ]; then
        DISTRIBUTION="Rocks"
elif [ -f /etc/SuSE-release ]; then
        DISTRIBUTION="SuSE"
elif [ -f /etc/debian_version ]; then
        DISTRIBUTION="Debian"
else
        DISTRIBUTION=$(ls /etc/*-release | head -n 1 | xargs -iXXX basename XXX -release 2> /dev/null)
fi


if [ -d /etc/sysconfig/network-scripts ]; then
    NETWORK_CONF_DIR="/etc/sysconfig/network-scripts"
elif [ -d /etc/sysconfig/network ]; then
    NETWORK_CONF_DIR="/etc/sysconfig/network"
else
    NETWORK_CONF_DIR="/etc/infiniband"
fi

# Check if quilt is installed
QUILT=$(/usr/bin/which quilt  2> /dev/null)
if [ -z ${QUILT} ] || [ ! -x ${QUILT} ]; then
        QUILT_PATCH=0
fi

verbs=n
mthca=n
ipath=n
ipoib=n
sdp=n
srp=n
srp_target=n
iser=n
rds=n

# Handle the case there is no makedepend...
export PATH=${PATH}:${installdir}
# COLOR SETTINGS
SETCOLOR_SUCCESS="echo -en \\033[1;34m"
SETCOLOR_FAILURE="echo -en \\033[1;31m"
SETCOLOR_WARNING="echo -en \\033[1;35m"
SETCOLOR_NORMAL="echo -en \\033[0;39m"

###############################################################################
#                          Parsing command line                               #
###############################################################################
while [ ! -z "$1" ]; do
        case "$1" in
                -t|--tmpdir)
                        tmpdir=$2
                        shift 2
                        ;;
                --kernel-version=*)
                        KVERSION=`expr "x$1" : 'x[^=]*=\(.*\)'`
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} --kernel-version ${KVERSION}"
                        shift
                        ;;
                -kver|--kernel-version*)
                        KVERSION=$2
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} --kernel-version ${KVERSION}"
                        shift 2
                        ;;
                --kernel-sources=*)
                        KSRC=`expr "x$1" : 'x[^=]*=\(.*\)'`
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} --kernel-sources ${KSRC}"
                        shift
                        ;;
                -k|--ksrc|--kernel-sources)
                        KSRC=$2
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} --kernel-sources ${KSRC}"
                        shift 2
                        ;;
                -libdir | --libdir)
                        LIBDIR=$2
                        WITH_PACKAGES_USER="${WITH_PACKAGES_USER} --libdir ${LIBDIR}"
                        shift 2
                        ;;
                -libdir=* | --libdir=*)
                        LIBDIR=`expr "x$1" : 'x[^=]*=\(.*\)'`
                        WITH_PACKAGES_USER="${WITH_PACKAGES_USER} --libdir ${LIBDIR}"
                        shift
                        ;;
                -h|--help)
                        usage
                        shift
                        exit 0
                        ;;
                -b|--batch)
                        batch=yes
                        shift
                        ;;
                -c|--clean)
                        clean=yes
                        shift
                        ;;
                --nosu)
                        nosu=yes
                        shift
                        ;;
                --build_root)
                        BUILD_ROOT=$2
                        shift 2
                        ;;
                --build_only)
                        SET_ENV=0
                        shift
                        ;;
                --memtrack|--with-memtrack)
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} --with-memtrack"
                        shift
                        ;;
                --debug)
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} --with-ipoib_debug_data-mod --with-sdp_debug_data-mod"
                        shift
                        ;;
                -p|--prefix)
                        prefix=$2
                        shift 2
                        ;;
                --packages)
                        shift
                        BUILD_MODULES=1
                        while [ "$1" != "--" ]
                        do
                            PACKAGES="$PACKAGES $1"
                            shift
                        done    
                        if [ -z "$PACKAGES" ]; then
                                BUILD_MODULES=0
                        fi
                        shift
                        ;;
                --without-user)
                        INSTALL_USER=0
                        shift
                        ;;
                --without-kernel)
                        INSTALL_KERNEL=0
                        shift
                        ;;
                --with-all-libs)
                        WITH_PACKAGES_USER="${WITH_PACKAGES_USER} --with-all-libs"
                        LIBSDP=1
                        shift
                        ;;
                --without-modprobe)
                        MODPROBE_UPDATE=0
                        shift
                        ;;
                --with*-mod)
                        WITH_PACKAGES_KERNEL="${WITH_PACKAGES_KERNEL} $1"
                        BUILD_MODULES=1
                        shift
                        ;;
                --with*)
                        WITH_PACKAGES_USER="${WITH_PACKAGES_USER} $1"
                        case "$1" in
                                --with-libsdp)
                                        LIBSDP=1
                                ;;
                        esac
                        shift
                        ;;
                --all)
                        PACKAGES=$ALL_PACKAGES
                        BUILD_MODULES=1
                        shift
                        ;;
                *)
                        echo "Wrong parameter error: $1"
                        usage
                        exit 1
                        ;;
        esac
done

if [ "$nosu" == "no" ]; then
        if [ $UID -ne 0 ] && [ $SET_ENV -eq 1 ]; then
                echo You must be root to install
                exit 1
        fi
fi

# Set default value for PACKAGES  
if [ -z "$PACKAGES" ] && [ $BUILD_MODULES -eq 1 ]; then
        PACKAGES=$ALL_PACKAGES
fi

###############################################################################
#                          Global environment                                 #
###############################################################################

HOST_INFO=${tmpdir}/host_info.log
DEBUG_INFO=${tmpdir}/debug_info.tgz
KVERSION=${KVERSION:-$(uname -r)}
KVERSION_TWO_DIGITS=2$(echo ${KVERSION} | cut -d'.' -f2)
KSRC=${KSRC:-/lib/modules/${KVERSION}/build}
KERNEL_TREE_ROOT=${KSRC%*/*}
MAKE_KERNEL_DIR=${KSRC}
BUILD_IB_KERNEL_HOME=${tmpdir}/infiniband
IB_KERNEL_HOME=${KSRC}/drivers/infiniband

LIBDIR32="${prefix}/lib"
LIBDIR64="${prefix}/lib64"

case ${ARCH} in
        i?86|ia64)
        LIBDIR=${LIBDIR:-"${LIBDIR32}"}
        ;;
        *)
        LIBDIR=${LIBDIR:-"${LIBDIR64}"}
        ;;
esac

# make sure we have the tmpdir
/bin/rm -rf ${tmpdir}
mkdir -p ${tmpdir}

MODULES_CONF_PARAMS=""
WITH_MAKE_PARAMS=""

###############################################################################
#                          Global functions                                   #
###############################################################################

err_echo()
{
        echo
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_FAILURE
        echo "ERROR: $@" 
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
        echo "ERROR: $@" >> $LOGFILE
}

pass_echo()
{
        echo
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_SUCCESS
        echo "$@"
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
        echo  "$@" >> $LOGFILE
}

# Create debug info
get_debug_info()
{
        /bin/rm -f $HOST_INFO
        touch $HOST_INFO
        echo "Hostname: `hostname -s`" >> $HOST_INFO
        if [ -e ${installdir}/BUILD_ID ]; then 
                echo "openib BUILD_ID=`cat ${installdir}/BUILD_ID`" >> $HOST_INFO
        fi
        test -e /etc/issue && echo "OS: `cat /etc/issue`" >> $HOST_INFO
        echo "Current kernel: `uname -r`" >> $HOST_INFO
        echo "Building modules for kernel: ${KVERSION}" >> $HOST_INFO
        echo "Architecture: `uname -m`" >> $HOST_INFO
        echo "GCC version: `gcc --version`"  >> $HOST_INFO
        echo "CPU: `cat /proc/cpuinfo | /bin/grep -E \"model name|arch\" | head -1`" >> $HOST_INFO
        echo "`cat /proc/meminfo | /bin/grep \"MemTotal\"`" >> $HOST_INFO
        echo "Chipset: `/sbin/lspci 2> /dev/null | head -1 | cut -d ':' -f 2-`" >> $HOST_INFO
        echo >> $HOST_INFO
        echo "############# LSPCI ##############" >> $HOST_INFO
        /sbin/lspci 2> /dev/null >> $HOST_INFO
        echo >> $HOST_INFO
        echo "############# LSPCI -N ##############" >> $HOST_INFO
        /sbin/lspci -n 2> /dev/null >> $HOST_INFO

        echo                                 
        cd ${tmpdir}
        tar czf ${DEBUG_INFO} *.log
        echo "Please open an issue in the http://openib.org/bugzilla and attach $DEBUG_INFO"
}

# Execute and print command. Put output into the LOGFILE. Check exit status.
ex()
{
        echo "Running: $@" >> $LOGFILE
        eval $@ >> $LOGFILE 2>&1
        if [ $? -ne 0 ]; then
                err_echo "Failed to execute: $@"
                echo "See $LOGFILE for more details"
                echo
                get_debug_info
                exit 1
        fi
}


check_prev_install()
{
        # Check prefix and kernel version using info file:
        INFO=/etc/infiniband/info
        if [ -x ${INFO} ]; then
                info_prefix=$(${INFO} | grep -w prefix | cut -d '=' -f 2)
                if [ -x ${info_prefix}/sbin/ofed_uninstall.sh ] \
                        || [ -x ${info_prefix}/uninstall_gen2.sh ] \
                        || [ -x ${info_prefix}/uninstall.sh ]; then
                        if [ "$batch" == "no" ]; then
                                echo -n "Remove currently installed components (${info_prefix}/uninstall.sh) (y/n) [y] ? "
                                read force
                        else
                                force="yes"
                        fi
                        if [ "x$force" == "xy" ] \
                                || [ "x$force" == "xyes" ] \
                                || [ "x$force" == "xY" ] \
                                || [ "x$force" == "xYES" ] \
                                || [ "x$force" == "x" ]  ; then
                                echo "Clean old distribution at ${info_prefix}"
                                echo "Running: yes | ${info_prefix}/uninstall.sh" >> $LOGFILE
                                # Save ${tmpdir} from uninstall
                                [ -d ${tmpdir} ] && mv ${tmpdir} ${tmpdir}.save
                                yes | ${info_prefix}/sbin/ofed_uninstall.sh > /dev/null 2>&1 || \
                                yes | ${info_prefix}/uninstall*.sh > /dev/null 2>&1
                                [ -d ${tmpdir}.save ] && mv ${tmpdir}.save ${tmpdir}
                        fi

                fi
        fi

        # Uninstall vapi if exist
        if [ -d /usr/mellanox ]; then
                echo "Found /usr/mellanox!"
                if [ "$batch" == "no" ]; then
                        if test -x /usr/mellanox/uninstall.sh; then
                                echo -n "Remove currently installed components (/usr/mellanox/uninstall.sh) (y/n) [y] ? "
                                read force
                        else
                                force="no"
                        fi
                else
                        force="yes"
                fi

                if [ "x$force" == "xy" ] \
                        || [ "x$force" == "xyes" ] \
                        || [ "x$force" == "xY" ] \
                        || [ "x$force" == "xYES" ] \
                        || [ "x$force" == "x" ]  ; then
                        echo "Clean old distribution at /usr/mellanox"
                        echo "Running: yes | /usr/mellanox/uninstall.sh" >> $LOGFILE
                        yes | /usr/mellanox/uninstall.sh >> $LOGFILE 2>&1

                        if [ $? -ne 0 ]; then
                                if [ "$batch" == "no" ]; then
                                        err_echo "/usr/mellanox/uninstall.sh failed"
                                        exit 1
                                else
                                        if [ "${prefix}" == "/usr/mellanox" ]; then
                                                echo "Forcing uninstall (rm -rf ${prefix}):"
                                                ex "/bin/rm -rf ${prefix}"
                                        else
                                                echo "WARNING: Leaving /usr/mellanox. \
                                                Continuing installation..."
                                        fi
                                fi
                        fi
                fi

        fi

}


set_deps()
{
        for pack in $PACKAGES
        do
                        case $pack in
                        ib_verbs)
                                verbs=y
                        ;;
                        ib_mthca)
                                verbs=y
                                mthca=y
                        ;;
                        ib_ipath)
                                case ${KVERSION} in
                                        2.6.5-7.244*)
                                        echo "IPATH is not supported for this kernel"
                                        ;;
                                        *)
                                        verbs=y
                                        ipath=y
                                        ;;
                                esac
                        ;;
                        ib_ipoib)
                                ipoib=y
                        ;;
                        ib_sdp)
                                ipoib=y
                                sdp=y
                        ;;
                        ib_srp)
                                ipoib=y
                                srp=y
                        ;;
                        ib_srp_target)
                                ipoib=y
                                srp_target=y
                        ;;
                        ib_iser)
                                case ${KVERSION} in
                                        2.6.1[6-9]*)
                                        iser=y
                                        ;;
                                        *)
                                        echo "ISER is not supported for this kernel"
                                        ;;
                                esac
                        ;;
                        rds)
                                case ${KVERSION} in
                                        2.6.5-7.244*)
                                        echo "RDS is not supported for this kernel"
                                        ;;
                                        *)
                                        ipoib=y
                                        rds=y
                                        ;;
                                esac
                        ;;
                        *)
                                echo "Wrong package name: $pack"
                                usage
                                exit 2
                        ;;    
                
                esac
        done

        if [ "$verbs" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-core-mod"
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-user_mad-mod"
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-user_access-mod"
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-addr_trans-mod"
        fi

        if [ "$mthca" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-mthca-mod"
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-mthca_debug-mod"
        fi

        if [ "$ipath" == "y" ]; then

                case $(uname -m) in
                        i?86)
                                echo "IPATH is not supported on x86 architecture"
                                exit 1
                        ;;
                        *)
                        ;;
                esac

                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-ipath_inf-mod"
        fi

        if [ "$ehca" == "y" ]; then
                case $(uname -m) in
                        ppc|ppc64)
                                case ${KVERSION} in
                                        2.6.1[6-9]*)
                                        WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-ehca-mod"
                                        ;;
                                        *)
                                        echo "EHCA is not supported by ${KVERSION} kernel"
                                        ;;
                                esac
                        ;;
                        *)
                                echo "EHCA is not supported on $(uname -m) architecture"
                                exit 1
                        ;;
                esac

        fi
   
        if [ "$ipoib" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-ipoib-mod"
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-ipoib_debug-mod"
        fi
    
        if [ "$sdp" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-sdp-mod"
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-sdp_debug-mod"
        fi
    
        if [ "$srp" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-srp-mod"
        fi
     
        if [ "$iser" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-iser-mod"
        fi
     
        if [ "$rds" == "y" ]; then
                WITH_PACKAGES_KERNEL="$WITH_PACKAGES_KERNEL --with-rds-mod"
        fi
}


print_selected_packages()
{
        echo
        echo "The following kernel modules will be installed:"
        echo
        [ "$verbs" == "y" ]             && echo ib_verbs
        [ "$mthca" == "y" ]             && echo ib_mthca
        [ "$ipath" == "y" ]             && echo ib_ipath
        [ "$ipoib" == "y" ]             && echo ib_ipoib
        [ "$sdp" == "y" ]               && echo ib_sdp
        [ "$srp" == "y" ]               && echo ib_srp
        [ "$srp_target" == "y" ]        && echo ib_srp_target
        [ "$iser" == "y" ]              && echo ib_iser
        [ "$rds" == "y" ]               && echo ib_rds
        
}

###############################################################################
#                          Create build tree                                  #
###############################################################################
create_build_tree()
{
        echo
        echo Creating build tree under ${tmpdir}
        ex mkdir -p ${tmpdir}/${package_name}
        cd ${installdir}
        for ff in *
        do
                if [ -d ${ff}/ ]; then 
                        ex /bin/cp -a ${ff}/ ${tmpdir}/${package_name}
                else
                        ex /bin/cp -a ${ff} ${tmpdir}/${package_name}
                fi
        done
}

count_ib_ports()
{
        local cnt=0
        local tmp_cnt=0
    
        tmp_cnt=$(/sbin/lspci -n 2> /dev/null | grep "15b3:6282" | wc -l | tr -d '[:space:]') # Arbel mode
        cnt=$[ $cnt + 2*${tmp_cnt} ]
    
        tmp_cnt=$(/sbin/lspci -n 2> /dev/null | grep -E "15b3:5e8c|15b3:6274" | wc -l | tr -d '[:space:]') # Sinai
        cnt=$[ $cnt + ${tmp_cnt} ]

        tmp_cnt=$(/sbin/lspci -n 2> /dev/null | grep -E "15b3:5a44|15b3:6278" | wc -l | tr -d '[:space:]') # Tavor mode
        cnt=$[ $cnt + 2*${tmp_cnt} ]
    
        return $cnt
}

ipoib_config()
{
        # Network configuration
        INTERFACES=`/sbin/ifconfig | grep eth | awk '{print$1}' | sort -n | tr -s '\n' '\ '`
        for INTERFACE in $INTERFACES
        do
                ADDRS=`/sbin/ifconfig $INTERFACE | grep "inet addr"`
                if [[ -n $ADDRS ]]; then
                        LAN_INTERFACE=$INTERFACE
                        break
                fi
        done

        LAN_INTERFACE=${LAN_INTERFACE:-eth0}


        count_ib_ports
        ports_num=$?

        DEVICES=""
                           
        for (( i=0 ; i < $ports_num ; i++ ))
        do
                DEVICES="$DEVICES ib${i}"
        done    
        
        # Set default number of devices in case that no IB device currently installed on the system
        [ "$DEVICES" == "" ] && DEVICES="ib0 ib1"

        high_IP=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*inet addr:\([0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        let high_IP++
        low_IP=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*inet addr:[0-9]*\.\([0-9]*\.[0-9]*\.[0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        IPADDR=${high_IP}.${low_IP}

        NETMASK=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*Mask:\([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        low_Bcast=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*Bcast:[0-9]*\.\([0-9]*\.[0-9]*\.[0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        BROADCAST=${high_IP}.${low_Bcast}
        ONBOOT=1
        case "$NETMASK" in
                "255.255.255.0")
                        NETWORK=${IPADDR%*.*}.0
                ;;
                "255.255.0.0")
                        NETWORK=${IPADDR%*.*.*}.0.0
                ;;
                "255.0.0.0")
                        NETWORK=${IPADDR%*.*.*.*}.0.0.0
                ;;
                *)
                        NETWORK=${IPADDR%*.*.*.*}.0.0.0
                ;;
        esac

        IP_A=${IPADDR%*.*.*.*}
        IP_BCD=${IPADDR#*.}
        NET_BCD=${NETWORK#*.}
        BR_BCD=${BROADCAST#*.}

        if [ "$DISTRIBUTION" == "Debian" ]; then
                target=/etc/network/interfaces
                perl -ni -e 'if (/\#\#\# IPoIB \#\#\#$/) { $filter = 1 }' -e 'if (!$filter) { print }' -e 'if (/\#\#\# END IPoIB \#\#\#$/){ $filter = 0 }' $target
                echo >> $target
                echo "### IPoIB ###" >> $target
        fi

        for DEVICE in $DEVICES
        do
                target=${NETWORK_CONF_DIR}/ifcfg-${DEVICE}
                cat $NULL > $target

                if [ "$DISTRIBUTION" == "SuSE" ]; then
                        echo BOOTPROTO=\'static\' >> $target
                        echo IPADDR=\'${IP_A}.${IP_BCD}\' >> $target
                        echo NETMASK=\'$NETMASK\' >> $target
                        echo NETWORK=\'${IP_A}.${NET_BCD}\' >> $target
                        echo BROADCAST=\'${IP_A}.${BR_BCD}\' >> $target
                        echo REMOTE_IPADDR=\'\' >> $target
                        if [ "$ONBOOT" == "1" ]; then
                                echo STARTMODE=\'onboot\' >> $target
                        else
                                echo STARTMODE=\'manual\' >> $target
                        fi
                        echo WIRELESS=\'no\' >> $target
                elif [ "$DISTRIBUTION" == "Debian" ]; then
                        target=/etc/network/interfaces
                        echo >> $target
                        echo "auto ${DEVICE}" >> $target
                        echo "iface ${DEVICE} inet static" >> $target
                        echo "        address ${IP_A}.${IP_BCD}" >> $target
                        echo "        netmask $NETMASK" >> $target
                        echo "        network ${IP_A}.${NET_BCD}" >> $target
                        echo "        broadcast ${IP_A}.${BR_BCD}" >> $target
                else
                        echo DEVICE=$DEVICE >> $target
                        echo BOOTPROTO=static >> $target
                        echo IPADDR=${IP_A}.${IP_BCD} >> $target
                        echo NETMASK=$NETMASK >> $target
                        echo NETWORK=${IP_A}.${NET_BCD} >> $target
                        echo BROADCAST=${IP_A}.${BR_BCD} >> $target
                        if [ "$ONBOOT" == "1" ]; then
                                echo ONBOOT=yes >> $target
                        else
                                echo ONBOOT=no >> $target
                        fi
                fi

                let IP_A++

        done

        if [ "$DISTRIBUTION" == "Debian" ]; then
                echo "### END IPoIB ###" >> $target
        fi
}

###############################################################################
#                          MAIN area                                          #
###############################################################################
main()
{
        if [ "${KVERSION_TWO_DIGITS}" == "24" ]; then
                printf "\n${KVERSION} kernel is not supported by this package\n"
                exit 1
        fi

        if [ -e ${installdir}/BUILD_ID ]; then
            echo "BUILD_ID: $(cat ${installdir}/BUILD_ID)"
        fi


        if [ $SET_ENV -eq 1 ]; then
                LOGFILE=${tmpdir}/check_prev_install.log
                check_prev_install
        fi
    
        LOGFILE=${tmpdir}/pre_install.log

        set_deps
    
        if [ $INSTALL_KERNEL -ne 0 ]; then
                print_selected_packages
        fi
    
        if [ ! -d ${BUILD_ROOT}/etc ]; then
                mkdir -p ${BUILD_ROOT}/etc
        fi

                      
        LOGFILE=${tmpdir}/create_build_tree.log
        create_build_tree

        if [ $INSTALL_USER -ne 0 ]; then
            
                if [ -e ${installdir}/uninstall.sh ] && [ $SET_ENV -eq 1 ]; then
                        mkdir -p ${BUILD_ROOT}/${prefix}
                        cp -a ${installdir}/uninstall.sh ${BUILD_ROOT}/${prefix}/uninstall_gen2.sh
                fi
                LOGFILE=${tmpdir}/configure.user.log
                cd ${tmpdir}/${package_name}/${package_name_user}
                printf "\n Running configure for userspace ..."

                case ${ARCH} in
                        ppc64)
                            CONFIG_PARAMS='CFLAGS="$CFLAGS -g -O2 -m64" CXXFLAGS="$CXXFLAGS -g -O2 -m64" FFLAGS="$FFLAGS -g -O2 -m64" LDFLAGS="$LDFLAGS -g -O2 -m64 -L/usr/lib64" LDLIBS="$LDLIBS -m64 -g -O2 -L/usr/lib64"'
                        ;;
                esac

                ex $CONFIG_PARAMS ./configure --prefix=${prefix} --libdir=${LIBDIR} ${WITH_PACKAGES_USER}
                printf "\r\t\t\t\t\t (done)"
        fi

        if [ $INSTALL_KERNEL -ne 0 ]; then
                LOGFILE=${tmpdir}/configure.kernel.log
                cd ${tmpdir}/${package_name}/${package_name_kernel}
                printf "\n Running configure for kernel ..."
                ex ./configure --prefix=${prefix} ${WITH_PACKAGES_KERNEL}
                printf "\r\t\t\t\t\t (done)"

                if [ $BUILD_MODULES -eq 1 ]; then
                        LOGFILE=${tmpdir}/make_kernel.log
                        cd ${tmpdir}/${package_name}/${package_name_kernel}
                        printf "\n Running make kernel ..."
                        ex make kernel
                        printf "\r\t\t\t\t\t (done)"
                fi
        fi

        if [ $INSTALL_USER -ne 0 ]; then
                LOGFILE=${tmpdir}/make_user.log
                cd ${tmpdir}/${package_name}/${package_name_user}
                printf "\n Running make user ..."
                ex env $CONFIG_PARAMS make user
                printf "\r\t\t\t\t\t (done)"
        fi

        if [ $INSTALL_KERNEL -ne 0 ]; then
                if [ $BUILD_MODULES -eq 1 ]; then
                        if [ $SET_ENV -eq 1 ]; then
                                LOGFILE=${tmpdir}/make_install_kernel.log
                                printf "\n Running make install_kernel ..."
                                cd ${tmpdir}/${package_name}/${package_name_kernel}
                                ex make install_kernel MODULES_DIR=/lib/modules/${KVERSION}/updates
                                printf "\r\t\t\t\t\t (done)"
                        else
                                LOGFILE=${tmpdir}/make_install_kernel.log
                                printf "\n Running make install_kernel ..."
                                cd ${tmpdir}/${package_name}/${package_name_kernel}
                                ex make install_kernel MODULES_DIR=/lib/modules/${KVERSION}/updates DESTDIR=${BUILD_ROOT}
                                printf "\r\t\t\t\t\t (done)"
                        fi
                fi
        fi

        if [ $INSTALL_USER -ne 0 ]; then
                LOGFILE=${tmpdir}/make_install_user.log
                printf "\n Running make install_user ..."
                cd ${tmpdir}/${package_name}/${package_name_user}
                ex make DESTDIR=${BUILD_ROOT} install_user 
                printf "\r\t\t\t\t\t (done)"

                if [ $LIBSDP -eq 1 ]; then
                        if [ "${ARCH}" == "x86_64" ] || [ "${ARCH}" == "ppc64" ]; then
                                # Making OFED libsdp in ${tmpdir} (32 bit version )
                                LOGFILE=${tmpdir}/libsdp32.log
                                cd ${tmpdir}/${package_name}/${package_name_user}/src/userspace/libsdp
                                make clean >> ${LOGFILE} 2>&1 && 
                                ./configure --prefix=${prefix} --libdir=${LIBDIR32} --disable-libcheck \
                                CFLAGS="-g -O2 -m32" CXXFLAGS="-g -O2 -m32" FFLAGS="-g -O2 -m32" >> ${LOGFILE} 2>&1 && 
                                make >> ${LOGFILE} 2>&1 && 
                                make DESTDIR=${BUILD_ROOT} install >> ${LOGFILE} 2>&1
                                if [ $? -ne 0 ]; then
                                        echo
                                        echo "The compilation of libsdp 32 bit version failed. For more information look at: ${LOGFILE}"
                                        echo "Skipping libsdp 32 bit version..."
                                fi

                        fi
                fi
        fi

        LOGFILE=${tmpdir}/make_install_user.log
        # Add info file to include installation information                                  
        [ ! -d ${BUILD_ROOT}${IB_CONF_DIR} ] && mkdir -p ${BUILD_ROOT}${IB_CONF_DIR}
        [ ! -d ${BUILD_ROOT}${NETWORK_CONF_DIR} ] && mkdir -p ${BUILD_ROOT}${NETWORK_CONF_DIR}

        INFO=${BUILD_ROOT}${IB_CONF_DIR}/info
        /bin/rm -f ${INFO}
        touch ${INFO}

cat >> ${INFO} << EOFINFO
#!/bin/bash
EOFINFO

if [ $INSTALL_USER -ne 0 ]; then
cat >> ${INFO} << EOFINFO
echo prefix=${prefix}
echo
echo "User level: ${WITH_PACKAGES_USER}"
EOFINFO
fi

if [ $INSTALL_KERNEL -ne 0 ]; then
cat >> ${INFO} << EOFINFO
echo
echo Kernel=${KVERSION}
echo "Kernel level: ${WITH_PACKAGES_KERNEL}"
echo
EOFINFO
fi

chmod +x ${INFO} > /dev/null 2>&1


        
        # Run this part to update running environment
        if [ $SET_ENV -eq 1 ]; then

                mkdir -p ${BUILD_ROOT}/${prefix}
                if [ -e ${installdir}/BUILD_ID ]; then
                    cp -f ${installdir}/BUILD_ID ${BUILD_ROOT}/${prefix}
                else
                        if [ -e ${installdir_user}/BUILD_ID ]; then
                            echo "User:" >> ${BUILD_ROOT}/${prefix}/BUILD_ID
                            cat ${installdir_user}/BUILD_ID >> ${BUILD_ROOT}/${prefix}/BUILD_ID
                        fi
                        if [ -e ${installdir_kernel}/BUILD_ID ]; then
                            echo "Kernel:" >> ${BUILD_ROOT}/${prefix}/BUILD_ID
                            cat ${installdir_kernel}/BUILD_ID >> ${BUILD_ROOT}/${prefix}/BUILD_ID
                        fi
                fi
                
                if [ -s ${prefix}/etc/dat.conf ]; then
                        # udapl configuration file
                        rm -f /etc/dat.conf
                        perl -ni -e  "s@/usr/lib@${LIBDIR}@;print" ${prefix}/etc/dat.conf
                        install -m 0644 ${prefix}/etc/dat.conf /etc/dat.conf
                        rm -f ${prefix}/etc/dat.conf
                fi
                
if [ $INSTALL_USER -ne 0 ]; then
mkdir -p /etc/profile.d
if [ -d /etc/profile.d ]; then
cat > /etc/profile.d/ofed.sh << EOF
if ! echo \${PATH} | grep -q ${prefix}/bin ; then
        PATH=\${PATH}:${prefix}/bin
fi
if ! echo \${PATH} | grep -q ${prefix}/sbin ; then
        PATH=\${PATH}:${prefix}/sbin
fi
if ! echo \${MANPATH} | grep -q ${prefix}/man ; then
        MANPATH=\${MANPATH}:${prefix}/man
fi
EOF
cat > /etc/profile.d/ofed.csh << EOF
if (\$?path) then
if ( "\${path}" !~ *${prefix}/bin* ) then
        set path = ( \$path ${prefix}/bin )
endif
if ( "\${path}" !~ *${prefix}/sbin* ) then
        set path = ( \$path ${prefix}/sbin )
endif
else
        set path = ( ${prefix}/bin ${prefix}/sbin )
endif
if (\$?MANPATH) then
if ( "\${MANPATH}" !~ *${prefix}/man* ) then
        set MANPATH = ( \$path ${prefix}/man )
endif
else
        set MANPATH = ( ${prefix}/man )
endif
EOF
fi

                # Update /etc/ld.so.conf depending on architecture                                                     
        
                mkdir -p /etc/ld.so.conf.d
                if [ -d  ${LIBDIR32} ]; then
                        echo "${LIBDIR32}" > /etc/ld.so.conf.d/ofed.conf
                fi
        
                if [ -d  ${LIBDIR64} ]; then
                        echo "${LIBDIR64}" >> /etc/ld.so.conf.d/ofed.conf
                fi

                ex /sbin/ldconfig
fi

if [ $INSTALL_KERNEL -ne 0 ]; then

                printf "\n Running depmod ..."
                ex /sbin/depmod -r -ae ${KVERSION}
                printf "\r\t\t\t\t\t (done)"
        
                [ ! -d ${IB_CONF_DIR} ] && mkdir -p ${IB_CONF_DIR}

                install -m 0644 ${installdir}/${package_name_kernel}/ofed_scripts/openib.conf ${IB_CONF_DIR}

                if [ "$verbs" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load UCM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "UCM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDMA_CM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "RDMA_CM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDMA_UCM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "RDMA_UCM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Increase ib_mad thread priority" >> ${IB_CONF_DIR}/openib.conf
                        echo "RENICE_IB_MAD=no" >> ${IB_CONF_DIR}/openib.conf
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load UCM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# UCM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDMA_CM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# RDMA_CM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDMA_UCM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# RDMA_UCM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Increase ib_mad thread priority" >> ${IB_CONF_DIR}/openib.conf
                        echo "# RENICE_IB_MAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi

                if [ "$mthca" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load MTHCA module" >> ${IB_CONF_DIR}/openib.conf
                        echo "MTHCA_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load MTHCA module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# MTHCA_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                fi
 
                if [ "$ehca" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load EHCA module" >> ${IB_CONF_DIR}/openib.conf
                        echo "EHCA_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load EHCA module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# EHCA_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                fi
 
                if [ "$ipath" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load IPATH module" >> ${IB_CONF_DIR}/openib.conf
                        echo "IPATH_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load IPATH module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# IPATH_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                fi
                                                
                if [ "$ipoib" == "y" ]; then
                        printf "\n Configuring IPoIB..."
                        ipoib_config

                        echo >> ${IB_CONF_DIR}/openib.conf                                                
                        echo "# Load IPoIB" >> ${IB_CONF_DIR}/openib.conf
                        echo "IPOIB_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                        echo "# Set IPoIB connected mode" >> ${IB_CONF_DIR}/openib.conf
                        echo "SET_IPOIB_CM=yes" >> ${IB_CONF_DIR}/openib.conf
                        echo "IPOIB_MTU=65520" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf                                                
                        echo "# Enable IPoIB High Availability daemon" >> ${IB_CONF_DIR}/openib.conf
                        echo "IPOIBHA_ENABLE=no" >> ${IB_CONF_DIR}/openib.conf
                        echo "# PRIMARY_IPOIB_DEV=ib0" >> ${IB_CONF_DIR}/openib.conf
                        echo "# SECONDARY_IPOIB_DEV=ib1" >> ${IB_CONF_DIR}/openib.conf
                        printf "\r\t\t\t\t\t (done)"

if [ $MODPROBE_UPDATE -eq 1 ]; then
# case ${KVERSION} in
# 2.6.1[6-9]*)
# cat >> /etc/modprobe.conf << EOF
# options ib_mthca msi_x=1
# EOF
# ;;
# *)
# ;;
# esac

for (( i=0 ; i < $ports_num ; i++ ))
do
cat >> /etc/modprobe.conf << EOF
alias ib${i} ib_ipoib
EOF
done
fi

                else
                        echo >> ${IB_CONF_DIR}/openib.conf                                                
                        echo "# Load IPoIB" >> ${IB_CONF_DIR}/openib.conf
                        echo "# IPOIB_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$sdp" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SDP module" >> ${IB_CONF_DIR}/openib.conf
                        echo "SDP_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
if [ $MODPROBE_UPDATE -eq 1 ]; then
cat >> /etc/modprobe.conf << EOF
alias net-pf-27 ib_sdp
EOF
fi
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SDP module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# SDP_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$srp" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SRP initiator module" >> ${IB_CONF_DIR}/openib.conf
                        echo "SRP_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SRP initiator module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# SRP_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$srp_target" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SRP target module" >> ${IB_CONF_DIR}/openib.conf
                        echo "SRP_TARGET_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$iser" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load ISER module" >> ${IB_CONF_DIR}/openib.conf
                        echo "ISER_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                else
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load ISER module" >> ${IB_CONF_DIR}/openib.conf
                        echo "# ISER_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$rds" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDS module" >> ${IB_CONF_DIR}/openib.conf
                        echo "RDS_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi

                eval `grep udev_rules /etc/udev/udev.conf | grep -v '^#'`
                if [ ! -n "$udev_rules" ]; then
                    udev_rules="/etc/udev/rules.d/"
                fi

                if [ -d $udev_rules ]; then 
                    install -m 0644 ${installdir}/${package_name_kernel}/ofed_scripts/90-ib.rules ${udev_rules}
                    case "$(udevinfo -V | awk '{print $NF}' 2> /dev/null)" in
                        0[1-4]*)
                        sed -i -e 's/KERNEL==/KERNEL=/g'  ${udev_rules}/90-ib.rules
                        ;;
                    esac
                else
                    perl -ni -e 'if (/\# Infiniband devices \#$/) { $filter = 1 }' -e 'if (!$filter) { print }' -e 'if (/\# End Infiniband devices \#$/){ $filter = 0 }' /etc/udev/udev.rules
                    cat >> /etc/udev/udev.rules << EOF
# Infiniband devices #
KERNEL="umad*", NAME="infiniband/%k"
KERNEL="issm*", NAME="infiniband/%k"
KERNEL="ucm*", NAME="infiniband/%k", MODE="0666"
KERNEL="uverbs*", NAME="infiniband/%k", MODE="0666"
KERNEL="uat", NAME="infiniband/%k", MODE="0666"
KERNEL="ucma", NAME="infiniband/%k", MODE="0666"
KERNEL="rdma_cm", NAME="infiniband/%k", MODE="0666"
# End Infiniband devices #
EOF

                fi

                install -m 0755 ${installdir}/${package_name_kernel}/ofed_scripts/openibd /etc/init.d/
                if [ "$DISTRIBUTION" == "SuSE" ]; then
                        perl -i -ne 'if (m@^#!/bin/bash@) {
                        print q@#!/bin/bash
### BEGIN INIT INFO
#  Provides:       openibd
#  Required-Start: $local_fs $network
#  Required-Stop: opensmd
#  Default-Start:  2 3 5
#  Default-Stop: 0 1 2 6
#  Description:    Activates/Deactivates InfiniBand Driver to \
#                  start at boot time.
### END INIT INFO
@;
                 } else {
                     print;
                 }' /etc/init.d/openibd

                        /sbin/insserv openibd

                elif [ "$DISTRIBUTION" == "Debian" ]; then
                        perl -i -ne 'if (m@^#!/bin/bash@) {
                        print q@#!/bin/bash
### BEGIN INIT INFO
#  Provides:       openibd
#  Required-Start: $local_fs $network
#  Required-Stop: opensmd
#  Default-Start:  2 3 5
#  Default-Stop: 0 1 2 6
#  Description:    Activates/Deactivates InfiniBand Driver to \
#                  start at boot time.
### END INIT INFO
@;
                 } else {
                     print;
                 }' /etc/init.d/openibd

                        /usr/sbin/update-rc.d openibd defaults

                else
                        perl -i -ne 'if (m@^#!/bin/bash@) {
                        print q@#!/bin/bash
#
# Bring up/down openib
# 
#  chkconfig: 2345 15 85
#  description: Activates/Deactivates InfiniBand Driver to \
#               start at boot time.
# 
### BEGIN INIT INFO
# Provides:       openibd
### END INIT INFO
@;
                 } else {
                     print;
                 }' /etc/init.d/openibd

                        /sbin/chkconfig --del openibd
                        /sbin/chkconfig --add openibd

                fi

        fi
fi

if [ $SET_ENV -eq 1 ]; then
    installpkg=$(basename `readlink -f ${installdir}`)
    ex install -d ${prefix}/src/${installpkg}
    cd ${installdir}
    for ff in *
    do
            if [ -d ${ff}/ ]; then 
                    ex /bin/cp -a ${ff}/ ${prefix}/src/${installpkg}
            else
                    ex /bin/cp -a ${ff} ${prefix}/src/${installpkg}
            fi
    done
    ex install -d ${prefix}/src/openib/drivers/infiniband
    cp -a ${tmpdir}/${package_name}/${package_name_kernel}/drivers/infiniband/include/ ${prefix}/src/openib
    pushd ${prefix}/src/openib/drivers/infiniband > /dev/null 2>&1
    ln -snf ../../include
    popd > /dev/null 2>&1

    printf "\n Package sources installed under ${prefix}/src/$(basename ${installpkg})"
    printf "\n Include files installed under ${prefix}/src/openib/include"
fi
        printf "\n Installation finished successfully \n"

}

main
